/*
 * Decompiled with CFR 0.152.
 */
package com.grinderwolf.swm.plugin.loaders.mongo;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.grinderwolf.swm.api.exceptions.UnknownWorldException;
import com.grinderwolf.swm.api.exceptions.WorldInUseException;
import com.grinderwolf.swm.internal.com.mongodb.MongoException;
import com.grinderwolf.swm.internal.com.mongodb.MongoNamespace;
import com.grinderwolf.swm.internal.com.mongodb.client.MongoClient;
import com.grinderwolf.swm.internal.com.mongodb.client.MongoClients;
import com.grinderwolf.swm.internal.com.mongodb.client.MongoCollection;
import com.grinderwolf.swm.internal.com.mongodb.client.MongoCursor;
import com.grinderwolf.swm.internal.com.mongodb.client.MongoDatabase;
import com.grinderwolf.swm.internal.com.mongodb.client.gridfs.GridFSBucket;
import com.grinderwolf.swm.internal.com.mongodb.client.gridfs.GridFSBuckets;
import com.grinderwolf.swm.internal.com.mongodb.client.gridfs.model.GridFSFile;
import com.grinderwolf.swm.internal.com.mongodb.client.model.Filters;
import com.grinderwolf.swm.internal.com.mongodb.client.model.IndexOptions;
import com.grinderwolf.swm.internal.com.mongodb.client.model.Indexes;
import com.grinderwolf.swm.internal.com.mongodb.client.model.Updates;
import com.grinderwolf.swm.internal.com.mongodb.client.result.UpdateResult;
import com.grinderwolf.swm.internal.org.bson.Document;
import com.grinderwolf.swm.plugin.config.DatasourcesConfig;
import com.grinderwolf.swm.plugin.loaders.UpdatableLoader;
import com.grinderwolf.swm.plugin.log.Logging;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

public class MongoLoader
extends UpdatableLoader {
    private static final ScheduledExecutorService SERVICE = Executors.newScheduledThreadPool(2, new ThreadFactoryBuilder().setNameFormat("SWM MongoDB Lock Pool Thread #%1$d").build());
    private final Map<String, ScheduledFuture> lockedWorlds = new HashMap<String, ScheduledFuture>();
    private final MongoClient client;
    private final String database;
    private final String collection;

    public MongoLoader(DatasourcesConfig.MongoDBConfig config) throws MongoException {
        this.database = config.getDatabase();
        this.collection = config.getCollection();
        String authParams = !config.getUsername().isEmpty() && !config.getPassword().isEmpty() ? config.getUsername() + ":" + config.getPassword() + "@" : "";
        String authSource = !config.getAuthSource().isEmpty() ? "/?authSource=" + config.getAuthSource() : "";
        this.client = MongoClients.create("mongodb://" + authParams + config.getHost() + ":" + config.getPort() + authSource);
        MongoDatabase mongoDatabase = this.client.getDatabase(this.database);
        MongoCollection<Document> mongoCollection = mongoDatabase.getCollection(this.collection);
        mongoCollection.createIndex(Indexes.ascending("name"), new IndexOptions().unique(true));
    }

    @Override
    public void update() {
        MongoCollection<Document> mongoCollection;
        MongoCursor documents;
        MongoDatabase mongoDatabase = this.client.getDatabase(this.database);
        for (String collectionName : mongoDatabase.listCollectionNames()) {
            if (!collectionName.equals(this.collection + "_files.files") && !collectionName.equals(this.collection + "_files.chunks")) continue;
            Logging.info("Updating MongoDB database...");
            mongoDatabase.getCollection(this.collection + "_files.files").renameCollection(new MongoNamespace(this.database, this.collection + ".files"));
            mongoDatabase.getCollection(this.collection + "_files.chunks").renameCollection(new MongoNamespace(this.database, this.collection + ".chunks"));
            Logging.info("MongoDB database updated!");
            break;
        }
        if ((documents = (mongoCollection = mongoDatabase.getCollection(this.collection)).find(Filters.or(Filters.eq("locked", true), Filters.eq("locked", false))).cursor()).hasNext()) {
            Logging.warning("Your SWM MongoDB database is outdated. The update process will start in 10 seconds.");
            Logging.warning("Note that this update will make your database incompatible with older SWM versions.");
            Logging.warning("Make sure no other servers with older SWM versions are using this database.");
            Logging.warning("Shut down the server to prevent your database from being updated.");
            try {
                Thread.sleep(10000L);
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
            while (documents.hasNext()) {
                String worldName = ((Document)documents.next()).getString("name");
                mongoCollection.updateOne(Filters.eq("name", worldName), Updates.set("locked", 0L));
            }
        }
    }

    @Override
    public byte[] loadWorld(String worldName, boolean readOnly) throws UnknownWorldException, IOException, WorldInUseException {
        try {
            MongoDatabase mongoDatabase = this.client.getDatabase(this.database);
            MongoCollection<Document> mongoCollection = mongoDatabase.getCollection(this.collection);
            Document worldDoc = (Document)mongoCollection.find(Filters.eq("name", worldName)).first();
            if (worldDoc == null) {
                throw new UnknownWorldException(worldName);
            }
            if (!readOnly) {
                long lockedMillis = worldDoc.getLong("locked");
                if (System.currentTimeMillis() - lockedMillis <= 300000L) {
                    throw new WorldInUseException(worldName);
                }
                this.updateLock(worldName, true);
            }
            GridFSBucket bucket = GridFSBuckets.create(mongoDatabase, this.collection);
            ByteArrayOutputStream stream = new ByteArrayOutputStream();
            bucket.downloadToStream(worldName, (OutputStream)stream);
            return stream.toByteArray();
        }
        catch (MongoException ex) {
            throw new IOException(ex);
        }
    }

    private void updateLock(String worldName, boolean forceSchedule) {
        try {
            MongoDatabase mongoDatabase = this.client.getDatabase(this.database);
            MongoCollection<Document> mongoCollection = mongoDatabase.getCollection(this.collection);
            mongoCollection.updateOne(Filters.eq("name", worldName), Updates.set("locked", System.currentTimeMillis()));
        }
        catch (MongoException ex) {
            Logging.error("Failed to update the lock for world " + worldName + ":");
            ex.printStackTrace();
        }
        if (forceSchedule || this.lockedWorlds.containsKey(worldName)) {
            this.lockedWorlds.put(worldName, SERVICE.schedule(() -> this.updateLock(worldName, false), 60000L, TimeUnit.MILLISECONDS));
        }
    }

    @Override
    public boolean worldExists(String worldName) throws IOException {
        try {
            MongoDatabase mongoDatabase = this.client.getDatabase(this.database);
            MongoCollection<Document> mongoCollection = mongoDatabase.getCollection(this.collection);
            Document worldDoc = (Document)mongoCollection.find(Filters.eq("name", worldName)).first();
            return worldDoc != null;
        }
        catch (MongoException ex) {
            throw new IOException(ex);
        }
    }

    @Override
    public List<String> listWorlds() throws IOException {
        ArrayList<String> worldList = new ArrayList<String>();
        try {
            MongoDatabase mongoDatabase = this.client.getDatabase(this.database);
            MongoCollection<Document> mongoCollection = mongoDatabase.getCollection(this.collection);
            MongoCursor documents = mongoCollection.find().cursor();
            while (documents.hasNext()) {
                worldList.add(((Document)documents.next()).getString("name"));
            }
        }
        catch (MongoException ex) {
            throw new IOException(ex);
        }
        return worldList;
    }

    @Override
    public void saveWorld(String worldName, byte[] serializedWorld, boolean lock) throws IOException {
        try {
            long lockMillis;
            MongoDatabase mongoDatabase = this.client.getDatabase(this.database);
            GridFSBucket bucket = GridFSBuckets.create(mongoDatabase, this.collection);
            GridFSFile oldFile = (GridFSFile)bucket.find(Filters.eq("filename", worldName)).first();
            if (oldFile != null) {
                bucket.rename(oldFile.getObjectId(), worldName + "_backup");
            }
            bucket.uploadFromStream(worldName, new ByteArrayInputStream(serializedWorld));
            MongoCollection<Document> mongoCollection = mongoDatabase.getCollection(this.collection);
            Document worldDoc = (Document)mongoCollection.find(Filters.eq("name", worldName)).first();
            long l = lockMillis = lock ? System.currentTimeMillis() : 0L;
            if (worldDoc == null) {
                mongoCollection.insertOne(new Document().append("name", worldName).append("locked", lockMillis));
            } else if (System.currentTimeMillis() - worldDoc.getLong("locked") > 300000L && lock) {
                this.updateLock(worldName, true);
            }
        }
        catch (MongoException ex) {
            throw new IOException(ex);
        }
    }

    @Override
    public void unlockWorld(String worldName) throws IOException, UnknownWorldException {
        ScheduledFuture future = this.lockedWorlds.remove(worldName);
        if (future != null) {
            future.cancel(false);
        }
        try {
            MongoDatabase mongoDatabase = this.client.getDatabase(this.database);
            MongoCollection<Document> mongoCollection = mongoDatabase.getCollection(this.collection);
            UpdateResult result = mongoCollection.updateOne(Filters.eq("name", worldName), Updates.set("locked", 0L));
            if (result.getMatchedCount() == 0L) {
                throw new UnknownWorldException(worldName);
            }
        }
        catch (MongoException ex) {
            throw new IOException(ex);
        }
    }

    @Override
    public boolean isWorldLocked(String worldName) throws IOException, UnknownWorldException {
        if (this.lockedWorlds.containsKey(worldName)) {
            return true;
        }
        try {
            MongoDatabase mongoDatabase = this.client.getDatabase(this.database);
            MongoCollection<Document> mongoCollection = mongoDatabase.getCollection(this.collection);
            Document worldDoc = (Document)mongoCollection.find(Filters.eq("name", worldName)).first();
            if (worldDoc == null) {
                throw new UnknownWorldException(worldName);
            }
            return System.currentTimeMillis() - worldDoc.getLong("locked") <= 300000L;
        }
        catch (MongoException ex) {
            throw new IOException(ex);
        }
    }

    @Override
    public void deleteWorld(String worldName) throws IOException, UnknownWorldException {
        ScheduledFuture future = this.lockedWorlds.remove(worldName);
        if (future != null) {
            future.cancel(false);
        }
        try {
            MongoDatabase mongoDatabase = this.client.getDatabase(this.database);
            GridFSBucket bucket = GridFSBuckets.create(mongoDatabase, this.collection);
            GridFSFile file = (GridFSFile)bucket.find(Filters.eq("filename", worldName)).first();
            if (file == null) {
                throw new UnknownWorldException(worldName);
            }
            bucket.delete(file.getObjectId());
            file = (GridFSFile)bucket.find(Filters.eq("filename", worldName + "_backup")).first();
            if (file != null) {
                bucket.delete(file.getObjectId());
            }
            MongoCollection<Document> mongoCollection = mongoDatabase.getCollection(this.collection);
            mongoCollection.deleteOne(Filters.eq("name", worldName));
        }
        catch (MongoException ex) {
            throw new IOException(ex);
        }
    }
}

